<?php

namespace App\Http\Controllers;


use App\Models\Cases;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;
use Yajra\DataTables\DataTables;
use App\Exports\CaseExport;
use App\Exports\TemplateCaseExport;
use App\Imports\CaseImport;
use App\Exports\ErrorCaseExport;


class CaseController extends Controller
{

    public function __construct()
    {
    }

    public function index()
    {
        $access = $this->setAccess();
        $datas = $this->setDatas();
        return view('case.index', compact('datas', 'access'));
    }

    private function setAccess()
    {
        $user = auth()->user();
        return  [
            'canEdit' => $user->can('edit-case'),
            'canDelete' => $user->can('delete-case'),
            'canCreate' => $user->can('create-case'),
            'canDownload' => $user->can('download-case'),
            'canUpload' => $user->can('upload-case'),
            'canExport' => $user->can('export-case'),
        ];
    }

    private function setDatas()
    {
        $datas['title']  = 'Case';
        $datas['url_store']  = route('case.store');
        $datas['url_redirect']  = route('case.index');
        return $datas;
    }

    public function listdata()
    {
        $data = Cases::select('id', 'code', 'name')
            ->where('deleted_at', null)
            ->orderBy('id', 'asc')
            ->get();
        return DataTables::of($data)->make(true);
    }

    public function create()
    {
        $datas = $this->setDatas();
        $datas['title']  = 'Form Add';
        $data = new Cases();
        return view('case.form', compact('data', 'datas'));
    }

    public function edit($id)
    {
        $datas = $this->setDatas();
        $datas['title']  = 'Form Edit';
        $data = Cases::find($id);
        return view('case.form', compact('data', 'datas'));
    }


    public function store(Request $request)
    {
        $where = function ($query) {
            $query->whereNull('deleted_at');
        };

        if ($request->filled('id')) {
            $where = function ($query) use ($request) {
                $query->whereNull('deleted_at')
                    ->where('id', '<>', $request->id);
            };
        }

        $validate = $request->validate([
            'code' => [
                'required',
                Rule::unique('m_case', 'code')->where($where),
                'min:2',
                'max:50',
            ]
        ], [
            'code.required' => 'Column cannot be empty.',
            'code.unique' => "Code ({$request->input('code')}) already exists.",
            'code.min' => 'Minimum length is 2 characters.',
            'code.max' => 'Maximum length is 10 characters.'
        ]);

        $response = array(1002, 'nothing can be processed');
        try {
            $user = DB::table('users')->where('id', Auth::id())->first();
            if (isset($request->id)) {
                $Case = Cases::findOrFail($request->id);
                $Case->code = $request->code;
                // $Case->name = $request->name;
                $Case->updated_who = $user->name;
                $Case->save();
                $response = array(1003, 'updated successfully');
            } else {

                Cases::create([
                    'code' => $request->code,
                    // 'name' => $request->name,
                    'created_who' => $user->name
                ]);

                $response = array(1003, 'saved successfully');
            }
        } catch (Exception $e) {
            $response = array(1002, 'exception occurs : ' . $e->getMessage());
        }

        return json_encode($response);
    }

    public function show($id)
    {
        $data = Cases::find($id);
        $datas = $this->setDatas();
        return view('case.show', compact('data', 'datas'));
    }


    private function isCaseReferenced($id)
    {
        // Check if the case is referenced in m_mapping or m_mapping_detail tables
        $count = DB::table('t_mapping')
            ->where('case_id', $id)
            ->orWhereExists(function ($query) use ($id) {
                $query->select(DB::raw(1))
                    ->from('t_mapping_detail')
                    ->whereRaw('t_mapping_detail.mapping_id = t_mapping.id')
                    ->where('t_mapping.case_id', $id);
            })
            ->count();

        return $count > 0;
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $response = [];

        try {
            // Check if the case is referenced in other tables
            if ($this->isCaseReferenced($id)) {
                $response = array(1002, 'Data is still referenced in other tables and cannot be deleted.');
            } else {
                // Delete the case if not referenced 
                $Case = Cases::findOrFail($id);
                if ($Case) {
                    $Case->code = 'CNCL_' . $id . '_' . $Case->code;
                    $Case->deleted_at = date('Y-m-d H:i:s');
                    $Case->save();
                    $response = array(1003, 'Deleted successfully.');
                } else {
                    $response = array(1002, 'Data not found');
                }
            }
        } catch (\Exception $e) {
            $response = array(1002, 'exception occurs : ' . $e->getMessage());
        }

        return json_encode($response);
    }


    public function excelAll()
    {
        return Excel::download(new CaseExport, 'case.xlsx');
    }

    public function downloadTemplate()
    {
        return Excel::download(new TemplateCaseExport, 'case_template.xlsx');
    }

    public function showUploadForm()
    {
        $datas = $this->setDatas();
        return view('case.upload', compact('datas'));
    }

    // Method to handle file upload
    public function upload(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 422);
        }

        try {
            $file = $request->file('file');
            $import = new CaseImport();
            Excel::import($import, $file);

            // Ambil data error jika ada
            $errors = $import->getErrors();
            $dataError = $import->getDataErrors();
            $successCount = $import->getSuccessCount();
            $errorCount = count($errors);

            if ($errorCount > 0) {
                // Jika ada error, buat dan kembalikan URL untuk mengunduh file dengan error
                $error_case = new ErrorCaseExport($dataError, $errors);
                $filePath = 'error_case.xlsx'; // Tentukan nama file
                // Excel::store($error_case, $filePath, \Maatwebsite\Excel\Excel::XLSX);
                Excel::store($error_case, $filePath);


                return response()->json([
                    'message' => 'File processed with errors.',
                    'successCount' => $successCount,
                    'errorCount' => $errorCount,
                    'downloadUrl' => route('case.downloadError', ['file' => $filePath]),
                ], 200);
            }

            return response()->json([
                'message' => 'File processed successfully.',
                'successCount' => $successCount,
                'errorCount' => $errorCount,
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
