<?php

namespace App\Http\Controllers;


use App\Models\Destination;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;
use Yajra\DataTables\DataTables;
use App\Exports\DestinationExport;
use App\Exports\TemplateDestinationExport;
use App\Exports\ErrorDestinationExport;
use App\Imports\DestinationImport;


class DestinationController extends Controller
{

    public function __construct()
    {
    }

    public function index()
    {
        $access = $this->setAccess();
        $datas = $this->setDatas();
        return view('destinations.index', compact('datas', 'access'));
    }

    private function setAccess()
    {
        $user = auth()->user();
        return  [
            'canEdit' => $user->can('edit-destination'),
            'canDelete' => $user->can('delete-destination'),
            'canCreate' => $user->can('create-destination'),
            'canDownload' => $user->can('download-destination'),
            'canUpload' => $user->can('upload-destination'),
            'canExport' => $user->can('export-destination'),
        ];
    }

    private function setDatas()
    {
        $datas['title']  = 'Destination';
        $datas['url_store']  = route('destinations.store');
        $datas['url_redirect']  = route('destinations.index');
        return $datas;
    }

    public function listdata()
    {
        $data = Destination::select('id', 'code', 'name')
            ->where('deleted_at', null)
            ->orderBy('id', 'asc')
            ->get();
        return DataTables::of($data)->make(true);
    }

    public function create()
    {
        $datas = $this->setDatas();
        $datas['title']  = 'Form Create';
        $data = new Destination();
        return view('destinations.form', compact('data', 'datas'));
    }

    public function edit($id)
    {
        $datas = $this->setDatas();
        $datas['title']  = 'Form Edit';
        $data = Destination::find($id);
        return view('destinations.form', compact('data', 'datas'));
    }


    public function store(Request $request)
    {
        $where = function ($query) {
            $query->whereNull('deleted_at');
        };

        if ($request->filled('id')) {
            $where = function ($query) use ($request) {
                $query->whereNull('deleted_at')
                    ->where('id', '<>', $request->id);
            };
        }

        $validate = $request->validate([
            'code' => [
                'required',
                Rule::unique('m_destination', 'code')->where($where),
                'min:2',
                'max:50',
            ],
            'name' => 'required',
        ], [
            'code.required' => 'Column cannot be empty.',
            'code.unique' => "Code ({$request->input('code')}) already exists.",
            'code.min' => 'Minimum length is 2 characters.',
            'code.max' => 'Maximum length is 10 characters.',
            'name.required' => 'Column cannot be empty.',
        ]);

        $response = array(1002, 'nothing can be processed');
        try {
            $user = DB::table('users')->where('id', Auth::id())->first();
            if (isset($request->id)) {
                $destination = Destination::findOrFail($request->id);
                $destination->code = $request->code;
                $destination->name = $request->name;
                $destination->updated_who = $user->name;
                $destination->save();
                $response = array(1003, 'updated successfully');
            } else {
                Destination::create([
                    'code' => $request->code,
                    'name' => $request->name,
                    'created_who' => $user->name
                ]);

                $response = array(1003, 'saved successfully');
            }
        } catch (Exception $e) {
            $response = array(1002, 'exception occurs : ' . $e->getMessage());
        }

        return json_encode($response);
    }

    public function show($id)
    {
        $data = Destination::find($id);
        $datas = $this->setDatas();
        return view('destinations.show', compact('data', 'datas'));
    }


    private function isDestinationReferenced($id)
    {
        // Check if the destination is referenced in m_mapping or m_mapping_detail tables
        $count = DB::table('t_mapping')
            ->where('destination_id', $id)
            ->orWhereExists(function ($query) use ($id) {
                $query->select(DB::raw(1))
                    ->from('t_mapping_detail')
                    ->whereRaw('t_mapping_detail.mapping_id = t_mapping.id')
                    ->where('t_mapping.destination_id', $id);
            })
            ->count();

        return $count > 0;
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $response = [];

        try {
            // Check if the Destination is referenced in other tables
            if ($this->isDestinationReferenced($id)) {
                $response = array(1002, 'Data is still referenced in other tables and cannot be deleted.');
            } else {
                // Delete the Destination if not referenced 
                $Destination = Destinations::findOrFail($id);
                if ($Destination) {
                    $Destination->code = 'CNCL_' . $id . '_' . $Destination->code;
                    $Destination->deleted_at = date('Y-m-d H:i:s');
                    $Destination->save();
                    $response = array(1003, 'Deleted successfully.');
                } else {
                    $response = array(1002, 'Data not found');
                }
            }
        } catch (\Exception $e) {
            $response = array(1002, 'exception occurs : ' . $e->getMessage());
        }

        return json_encode($response);
    }


    public function excelAll()
    {
        return Excel::download(new DestinationExport, 'destinations.xlsx');
    }

    public function downloadTemplate()
    {
        return Excel::download(new TemplateDestinationExport, 'destination_template.xlsx');
    }

    public function showUploadForm()
    {
        $datas = $this->setDatas();
        return view('destinations.upload', compact('datas'));
    }

    // Method to handle file upload
    public function upload(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 422);
        }

        try {
            $file = $request->file('file');
            $import = new DestinationImport();
            Excel::import($import, $file);

            // Ambil data error jika ada
            $errors = $import->getErrors();
            $dataError = $import->getDataErrors();
            $successCount = $import->getSuccessCount();
            $errorCount = count($errors);

            if ($errorCount > 0) {
                // Jika ada error, buat dan kembalikan URL untuk mengunduh file dengan error
                $error_destination = new ErrorDestinationExport($dataError, $errors);
                $filePath = 'error_destination.xlsx'; // Tentukan nama file
                // Excel::store($error_destination, $filePath, \Maatwebsite\Excel\Excel::XLSX);
                Excel::store($error_destination, $filePath);


                return response()->json([
                    'message' => 'File processed with errors.',
                    'successCount' => $successCount,
                    'errorCount' => $errorCount,
                    'downloadUrl' => route('destinations.downloadError', ['file' => $filePath]),
                ], 200);
            }

            return response()->json([
                'message' => 'File processed successfully.',
                'successCount' => $successCount,
                'errorCount' => $errorCount,
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
