<?php

namespace App\Http\Controllers;


use App\Exports\ErrorItemExport;
use App\Models\Item;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;
use Yajra\DataTables\DataTables;
use App\Exports\ItemExports;
use App\Exports\TemplateItemExport;
use App\Imports\ItemImport;


class ItemController extends Controller
{

    public function __construct()
    {
    }

    public function index()
    {
        $access = $this->setAccess();
        $datas = $this->setDatas(); 
        return view('items.index', compact('datas', 'access'));
    }

    private function setAccess()
    {
        $user = auth()->user();
        return  [
            'canEdit' => $user->can('edit-items'),
            'canDelete' => $user->can('delete-items'),
            'canCreate' => $user->can('create-items'),
            'canDownload' => $user->can('download-items'),
            'canUpload' => $user->can('upload-items'),
            'canExport' => $user->can('export-items'),
        ];
    }

    private function setDatas()
    {
        $datas['title']  = 'Part Number';
        $datas['url_store']  = route('items.store');
        $datas['url_redirect']  = route('items.index');
        return $datas;
    }

    public function listdata()
    {
        $data = Item::select('id', 'code', 'name', 'address')
            ->where('deleted_at', null)
            ->orderBy('id', 'asc')
            ->get();
        return DataTables::of($data)->make(true);
    }

    public function create()
    {
        $datas = $this->setDatas();
        $datas['title']  = 'Form Add';
        $data = new Item();
        return view('items.form', compact('data', 'datas'));
    }

    public function edit($id)
    {
        $datas = $this->setDatas();
        $datas['title']  = 'Form Edit';
        $data = Item::find($id);
        return view('items.form', compact('data', 'datas'));
    }


    public function store(Request $request)
    {
        $where = function ($query) {
            $query->whereNull('deleted_at');
        };

        if ($request->filled('id')) {
            $where = function ($query) use ($request) {
                $query->whereNull('deleted_at')
                    ->where('id', '<>', $request->id);
            };
        }

        $validate = $request->validate([
            'code' => [
                'required',
                Rule::unique('m_item', 'code')->where($where),
                'min:2',
                'max:50',
            ],
            'name' => 'required',
        ], [
            'code.required' => 'Column cannot be empty.',
            'code.unique' => "Code ({$request->input('code')}) already exists.",
            'code.min' => 'Minimum length is 2 characters.',
            'code.max' => 'Maximum length is 10 characters.',
            'name.required' => 'Column cannot be empty.',
        ]);

        $response = array(1002, 'nothing can be processed');
        try {
            $user = DB::table('users')->where('id', Auth::id())->first();
            if (isset($request->id)) {
                $Item = Item::findOrFail($request->id);
                $Item->code = $request->code;
                $Item->name = $request->name;
                $Item->updated_who = $user->name;
                $Item->save();
                $response = array(1003, 'updated successfully');
            } else {

                Item::create([
                    'code' => $request->code,
                    'name' => $request->name,
                    'address' => $request->address,
                    'created_who' => $user->name
                ]);

                $response = array(1003, 'saved successfully');
            }
        } catch (Exception $e) {
            $response = array(1002, 'exception occurs : ' . $e->getMessage());
        }

        return json_encode($response);
    }

    public function show($id)
    {
        $data = Item::find($id);
        $datas = $this->setDatas();
        return view('items.show', compact('data', 'datas'));
    }


    private function isItemReferenced($id)
    {
        // Check if the items is referenced in m_mapping or m_mapping_detail tables
        $count = DB::table('t_mapping') 
            ->WhereExists(function ($query) use ($id) {
                $query->select(DB::raw(1))
                    ->from('t_mapping_detail')
                    ->whereRaw('t_mapping_detail.mapping_id = t_mapping.id')
                    ->where('t_mapping_detail.item_id', $id);
            })
            ->count();

        return $count > 0;
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $response = [];

        try {
            // Check if the Item is referenced in other tables
            if ($this->isItemReferenced($id)) {
                $response = array(1002, 'Data is still referenced in other tables and cannot be deleted.');
            } else {
                // Delete the Item if not referenced 
                $Item = Item::findOrFail($id);
                if ($Item) {
                    $Item->code = 'CNCL_' . $id . '_' . $Item->code;
                    $Item->deleted_at = date('Y-m-d H:i:s');
                    $Item->save();
                    $response = array(1003, 'Deleted successfully.');
                } else {
                    $response = array(1002, 'Data not found');
                }
            }
        } catch (\Exception $e) {
            $response = array(1002, 'exception occurs : ' . $e->getMessage());
        }

        return json_encode($response);
    }


    public function excelAll()
    {
        return Excel::download(new ItemExports, 'part_number.xlsx');
    }

    public function downloadTemplate()
    {
        return Excel::download(new TemplateItemExport, 'part_number_template.xlsx');
    }

    public function showUploadForm()
    {
        $datas = $this->setDatas();
        return view('items.upload', compact('datas'));
    }

    // Method to handle file upload
    public function upload(Request $request)
    {
        // Validasi input file
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 422);
        }

        try {
            // Ambil file dari request
            $file = $request->file('file');
            $import = new ItemImport();
            Excel::import($import, $file);

            // Ambil data kesalahan dan jumlah sukses
            $errors = $import->getErrors();
            $dataError = $import->getDataErrors();
            $successCount = $import->getSuccessCount();
            $errorCount = count($errors);

            if ($errorCount > 0) {
                // Jika ada kesalahan, buat objek ErrorItemExport
                $error_item = new ErrorItemExport($dataError, $errors);
                $filePath = 'error_part_number.xlsx'; // Tentukan nama file

                // Simpan file kesalahan
                Excel::store($error_item, $filePath);

                return response()->json([
                    'message' => 'File processed with errors.',
                    'successCount' => $successCount,
                    'errorCount' => $errorCount,
                    'downloadUrl' => route('items.downloadError', ['file' => $filePath]),
                ], 200);
            }

            return response()->json([
                'message' => 'File processed successfully.',
                'successCount' => $successCount,
                'errorCount' => $errorCount,
            ], 200);
        } catch (\Exception $e) {
            // Tangani kesalahan lainnya
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function fetchItemData($id)
    {
        $item = Item::find($id);
        if ($item) {
            return response()->json([
                'id' => $item->id,
                'code' => $item->code,
                'name' => $item->name,
                'address' => $item->address,
            ]);
        } else {
            return response()->json(null, 404);
        }
    }
}
