<?php

namespace App\Http\Controllers;


use App\Exports\LotProcessExport;
use App\Exports\TemplateLotProcess;
use App\Exports\ErrorLotProcessExport;
use App\Imports\LotProcessImport;
use App\Models\LotProcess;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;
use Yajra\DataTables\DataTables;


class LotProcessController extends Controller
{

    public function __construct()
    {
    }

    public function index()
    {
        $access = $this->setAccess();
        $datas = $this->setDatas();
        return view('lot-process.index', compact('datas', 'access'));
    }

    private function setAccess()
    {
        $user = auth()->user();
        return  [
            'canEdit' => $user->can('edit-lot-process'),
            'canDelete' => $user->can('delete-lot-process'),
            'canCreate' => $user->can('create-lot-process'),
            'canDownload' => $user->can('download-lot-process'),
            'canUpload' => $user->can('upload-lot-process'),
            'canExport' => $user->can('export-lot-process'),
        ];
    }

    private function setDatas()
    {
        $datas['title']  = 'Lot Process';
        $datas['url_store']  = route('lot-process.store');
        $datas['url_redirect']  = route('lot-process.index');
        return $datas;
    }

    public function listdata()
    {
        $data = LotProcess::select('id', 'code', 'name')
            ->where('deleted_at', null)
            ->orderBy('id', 'asc')
            ->get();
        return DataTables::of($data)->make(true);
    }

    public function create()
    {
        $datas = $this->setDatas();
        $datas['title']  = 'Form Add';
        $data = new LotProcess();
        return view('lot-process.form', compact('data', 'datas'));
    }

    public function edit($id)
    {
        $datas = $this->setDatas();
        $datas['title']  = 'Form Edit';
        $data = LotProcess::find($id);
        return view('lot-process.form', compact('data', 'datas'));
    }


    public function store(Request $request)
    {
        $where = function ($query) {
            $query->whereNull('deleted_at');
        };

        if ($request->filled('id')) {
            $where = function ($query) use ($request) {
                $query->whereNull('deleted_at')
                    ->where('id', '<>', $request->id);
            };
        }

        $validate = $request->validate([
            'code' => [
                'required',
                Rule::unique('m_lot_process', 'code')->where($where),
                'min:2',
                'max:50',
            ]
        ], [
            'code.required' => 'Column cannot be empty.',
            'code.unique' => "Code ({$request->input('code')}) already exists.",
            'code.min' => 'Minimum length is 2 characters.',
            'code.max' => 'Maximum length is 10 characters.',
        ]);

        $response = array(1002, 'nothing can be processed');
        try {
            $user = DB::table('users')->where('id', Auth::id())->first();
            if (isset($request->id)) {
                $LotProcess = LotProcess::findOrFail($request->id);
                $LotProcess->code = $request->code;
                // $LotProcess->name = $request->name;
                $LotProcess->updated_who = $user->name;
                $LotProcess->save();
                $response = array(1003, 'updated successfully');
            } else {

                LotProcess::create([
                    'code' => $request->code,
                    // 'name' => $request->name,
                    'created_who' => $user->name
                ]);

                $response = array(1003, 'saved successfully');
            }
        } catch (Exception $e) {
            $response = array(1002, 'exception occurs : ' . $e->getMessage());
        }

        return json_encode($response);
    }

    public function show($id)
    {
        $data = LotProcess::find($id);
        $datas = $this->setDatas();
        return view('lot-process.show', compact('data', 'datas'));
    }


    private function isLotProcessReferenced($id)
    {
        // Check if the lot-process is referenced in m_mapping or m_mapping_detail tables
        $count = DB::table('t_mapping')
            ->where('lot_process_id', $id)
            ->orWhereExists(function ($query) use ($id) {
                $query->select(DB::raw(1))
                    ->from('t_mapping_detail')
                    ->whereRaw('t_mapping_detail.mapping_id = t_mapping.id')
                    ->where('t_mapping.lot_process_id', $id);
            })
            ->count();

        return $count > 0;
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $response = [];

        try {
            // Check if the lot-process is referenced in other tables
            if ($this->isLotProcessReferenced($id)) {
                $response = array(1002, 'Data is still referenced in other tables and cannot be deleted.');
            } else {
                // Delete the lot-process if not referenced 
                $LotProcess = LotProcess::findOrFail($id);
                if ($LotProcess) {
                    $LotProcess->code = 'CNCL_' . $id . '_' . $LotProcess->code;
                    $LotProcess->deleted_at = date('Y-m-d H:i:s');
                    $LotProcess->save();
                    $response = array(1003, 'Deleted successfully.');
                } else {
                    $response = array(1002, 'Data not found');
                }
            }
        } catch (\Exception $e) {
            $response = array(1002, 'exception occurs : ' . $e->getMessage());
        }

        return json_encode($response);
    }


    public function excelAll()
    {
        return Excel::download(new LotProcessExport, 'lot-process.xlsx');
    }

    public function downloadTemplate()
    {
        return Excel::download(new TemplateLotProcess, 'lot_process_template.xlsx');
    }

    public function showUploadForm()
    {
        $datas = $this->setDatas();
        return view('lot-process.upload', compact('datas'));
    }

    // Method to handle file upload
    public function upload(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 422);
        }

        try {
            $file = $request->file('file');
            $import = new LotProcessImport();
            Excel::import($import, $file);

            // Ambil data error jika ada
            $errors = $import->getErrors();
            $dataError = $import->getDataErrors();
            $successCount = $import->getSuccessCount();
            $errorCount = count($errors);

            if ($errorCount > 0) {
                // Jika ada error, buat dan kembalikan URL untuk mengunduh file dengan error
                $error_lot_process = new ErrorLotProcessExport($dataError, $errors);
                $filePath = 'error_lot_process.xlsx'; // Tentukan nama file
                // Excel::store($error_lot_process, $filePath, \Maatwebsite\Excel\Excel::XLSX);
                Excel::store($error_lot_process, $filePath);


                return response()->json([
                    'message' => 'File processed with errors.',
                    'successCount' => $successCount,
                    'errorCount' => $errorCount,
                    'downloadUrl' => route('lot-process.downloadError', ['file' => $filePath]),
                ], 200);
            }

            return response()->json([
                'message' => 'File processed successfully.',
                'successCount' => $successCount,
                'errorCount' => $errorCount,
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
