<?php

namespace App\Http\Controllers;

use App\Exports\ErrorMappingExport;
use App\Models\Destination;
use App\Models\Cases;
use App\Models\LotProcess;
use App\Models\Item;
use App\Models\Mapping;
use App\Models\MappingDetail;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use Yajra\DataTables\DataTables;
use App\Exports\MappingExport;
use App\Exports\MappingTemplateExport;
use App\Http\Requests\MappingUploadRequest;
use App\Imports\MappingImport; 

class MappingController extends Controller
{
    public function __construct()
    {
        // $this->middleware('auth');
        // $this->middleware('permission:view-mapping', ['only' => ['index', 'show', 'listdata']]);
        // $this->middleware('permission:create-mapping', ['only' => ['create', 'store']]);
        // $this->middleware('permission:edit-mapping', ['only' => ['edit', 'update']]);
        // $this->middleware('permission:delete-mapping', ['only' => ['destroy']]);
    }

    public function index()
    {
        $access = $this->setAccess();
        $datas = $this->setDatas();
        return view('mapping.index', compact('datas', 'access'));
    }

    private function setAccess()
    {
        $user = auth()->user();
        return  [
            'canEdit' => $user->can('edit-mapping'),
            'canDelete' => $user->can('delete-mapping'),
            'canCreate' => $user->can('create-mapping'),
            'canDownload' => $user->can('download-mapping'),
            'canUpload' => $user->can('upload-mapping'),
            'canExport' => $user->can('export-mapping'),
        ];
    }

    private function setDatas()
    {
        $datas['destinations'] = DB::table('m_destination')
            ->select('id', DB::raw("CONCAT(code,' / ', name) as code"))
            ->whereNull('deleted_at')
            ->get();

        $datas['lot_processes'] = DB::table('m_lot_process')
            ->select('id', 'code', 'name')
            ->whereNull('deleted_at')
            ->get();

        $datas['cases'] = DB::table('m_case')
            ->select('id', 'code', 'name')
            ->whereNull('deleted_at')
            ->get();

        $datas['items'] = DB::table('m_item')
            ->select('id', DB::raw("CONCAT(code,' / ', name) as code"))
            ->whereNull('deleted_at')
            ->get();

        $datas['title']  = 'Mapping';
        $datas['url_store']  = route('mapping.store');
        $datas['url_redirect']  = route('mapping.index');
        return $datas;
    }

    public function listdata()
    {
        $data = Mapping::select(
            't_mapping.id',
            't_mapping.start_date',
            't_mapping.end_date',
            DB::raw("CONCAT(m_destination.code,' / ', m_destination.name) as destination_name"),
            'm_lot_process.code as lot_process_name',
            'm_case.code as case_name'
        )
            ->join('m_destination', 't_mapping.destination_id', '=', 'm_destination.id')
            ->join('m_lot_process', 't_mapping.lot_process_id', '=', 'm_lot_process.id')
            ->join('m_case', 't_mapping.case_id', '=', 'm_case.id')
            ->orderBy('t_mapping.id', 'asc')
            ->whereNull('t_mapping.deleted_at')
            ->get();

        return DataTables::of($data)->make(true);
    }

    public function create()
    {
        $datas = $this->setDatas();

        return view('mapping.form', compact('datas'));
    }


    public function store(Request $request)
    {
        // Validasi request
        $validatedData = $request->validate([
            'destination_id' => 'required',
            'lot_process_id' => 'required',
            'case_id' => 'required',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'details' => 'required|array',
            'details.*.item_id' => 'required',
            'details.*.sequence' => 'required',
        ], [
            'destination_id.required' => 'Please select Destination.',
            'lot_process_id.required' => 'Please select Lot Process.',
            'case_id.required' => 'Please select Case.',
            'start_date.required' => 'Please input start date.',
            'start_date.date' => 'The start date is not a valid date.',
            'end_date.required' => 'Please input end date.',
            'end_date.date' => 'The end date cannot be less than the start date.',
            'end_date.after' => 'End date must be after the start date.',
            'details.required' => 'Details are required.',
            'details.array' => 'Details must be an array.',
            'details.*.item_id.required' => 'Please select an item for all details.',
            'details.*.sequence.required' => 'Please enter a sequence for all details.',
        ]);

        // dd($request->details);
        foreach ($request->details as $key => $detail) {
            if (!isset($detail['id'])) { //jika gaada id nya maka inseert dan itu wajib isi iamge
                $rownotif = 'For part number "' . $detail['name'] . '" on line ' . $key . ', ';
                $request->validate([
                    "details.$key.image" => 'required|image|mimes:jpeg,png,jpg|max:2048',
                ], [
                    "details.$key.image.required" => " $rownotif Please upload an image for part number",
                    "details.$key.image.image" => " $rownotif The file must be an image.",
                    "details.$key.image.mimes" => " $rownotif The image must be a file of type: jpeg, png, jpg",
                    "details.$key.image.max" => " $rownotif The image may not be greater than 2048 kilobytes.",
                ]);
            }
        }

        $response = array(1002, 'nothing can be processed');
        try {
            // Get authenticated user
            $user = Auth::user();

            if ($request->filled('id')) {
                $mapping = Mapping::findOrFail($request->id);
                $mapping->updated_who = $user->name;
            } else {
                $mapping = new Mapping();
                $mapping->created_who = $user->name;
            }

            $mapping->destination_id = $validatedData['destination_id'];
            $mapping->lot_process_id = $validatedData['lot_process_id'];
            $mapping->case_id = $validatedData['case_id'];
            $mapping->start_date = $validatedData['start_date'];
            $mapping->end_date = $validatedData['end_date'];
            $mapping->save();

            $list_detail_bertahan = array();
            foreach ($validatedData['details'] as $detail) {

                $imagePath = null;
                // Handle image upload for each detail
                if (isset($detail['image'])) {
                    $image = $detail['image'];
                    $extension = $image->getClientOriginalExtension();
                    $filename = time() . '_' . $image->getClientOriginalName();
                    $destinationPath = public_path('images/mapping');

                    // Resize and save image
                    $img = \Intervention\Image\Facades\Image::make($image->getRealPath());
                    $img->resize(800, null, function ($constraint) {
                        $constraint->aspectRatio();
                    });
                    $img->save($destinationPath . '/' . $filename);

                    $imagePath = 'images/mapping/' . $filename;
                }

                if (isset($detail['id']) and !empty($detail['id'])) {
                    $mappingDetail = MappingDetail::findOrFail($detail['id']);
                    $mappingDetail->updated_who =  $user->name;

                    array_push($list_detail_bertahan, $detail['id']);
                } else {
                    // Create new mapping detail record
                    $mappingDetail = new MappingDetail();
                    $mappingDetail->created_who =  $user->name;
                }

                $mappingDetail->item_id =  $detail['item_id'];
                $mappingDetail->sequence =  $detail['sequence'];
                if ($imagePath) {
                    $mappingDetail->image =  $imagePath;
                }

                // Save mapping detail under mapping
                $mapping->details()->save($mappingDetail);
            }

            if ($list_detail_bertahan) {
                //Update semua detail yang hilang saat edit , function delete() di laravel cuma ngflag deleted_at saja. jadi line nya gak ke hapus
                MappingDetail::where('mapping_id', $request->id)->whereNotIn('id', $list_detail_bertahan)->delete();
            }

            if ($request->filled('id')) {
                $response = array(1003, 'updated successfully');
            } else {
                $response = array(1003, 'saved successfully');
            }
            // Commit transaction
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            $pesan = $e->getMessage();
            // Periksa apakah pesan mengandung "#keyUniqHeader#"
            if (strpos($pesan, "#keyUniqHeader#") !== false) {
                // Jika ditemukan, lakukan split atau operasi lainnya
                $split = explode("#keyUniqHeader#", $pesan);

                $pesan = $split[1];
            }

            $response = array(1002, 'exception occurs : ' . $pesan);
        }

        return json_encode($response);
    }


    public function show($id)
    {
        $data = Mapping::with(['details', 'destination', 'lotProcess', 'mappingCase'])
            ->findOrFail($id);

        return view('mapping.show', compact('data'));
    }

    public function edit($id)
    {
        $datas = $this->setDatas();
        $sql = "select 
                a.id,
                item_id,
                b.code ,
                b.name ,
                b.address ,
                a.sequence,
                a.image
                from t_mapping_detail a 
                join t_mapping a2 on a2.id=a.mapping_id and a.deleted_at is null
                join m_item b on b.id=a.item_id 
                where a2.id = $id";
        $data_details = DB::select($sql);
        // print_r('<pre>');
        // print_r($data);
        // exit;
        $data = Mapping::find($id);
        return view('mapping.form', compact('data', 'data_details', 'datas'));
    }


    public function destroy($id)
    {
        $mapping = Mapping::findOrFail($id);
        $mapping->delete();

        return response()->json(['message' => 'Mapping data deleted successfully.'], 200);
    }

    public function excelAll()
    {
        return Excel::download(new MappingExport(), 'Mapping.xlsx');
    }

    public function downloadTemplate()
    {
        return Excel::download(new MappingTemplateExport(), 'mapping_template.xlsx');
    }

    // public function uploadTemplate(MappingUploadRequest $request)
    // {
    //     Excel::import(new MappingImport, $request->file('file'));

    //     return back()->with('success', 'File uploaded successfully.');
    // }

    public function showUploadForm()
    {
        return view('mapping.upload');
    }

//    public function uploadTemplate(Request $request)
//    {
//        $request->validate([
//            'file' => 'required|mimes:xlsx'
//        ]);
//
//        Excel::import(new MappingImport, $request->file('file'));
//
//        return back()->with('success', 'File uploaded successfully.');
//    }

    public function import(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 422);
        }

        try {
            $file = $request->file('file');
            $import = new MappingImport();
            Excel::import($import, $file);

            $errors = $import->getErrors();
            $dataError = $import->getDataErrors();
            $successCount = $import->getSuccessCount();
            $dataErrorDetails = $import->getDataErrorDetails(); // Pastikan method ini ada
            $errorMessageDetails = $import->getErrorMessageDetails(); // Pastikan method ini ada
            $errorCount = count($errors);

            if ($errorCount > 0) {
                // Create the error report
                $errorExport = new ErrorMappingExport($dataError, $errors, $dataErrorDetails, $errorMessageDetails);
                $filePath = 'error_mapping.xlsx'; // Name of the error file
                Excel::store($errorExport, $filePath); // Save the error file

                return response()->json([
                    'message' => 'File processed with errors.',
                    'successCount' => $successCount,
                    'errorCount' => $errorCount,
                    'downloadUrl' => route('mapping.downloadError', ['file' => $filePath]), // Route to download error file
                ], 200);
            }

            return response()->json([
                'message' => 'File processed successfully.',
                'successCount' => $successCount,
                'errorCount' => $errorCount,
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }


    public function downloadError($file)
    {
        $path = storage_path('app/' . $file);

        if (file_exists($path)) {
            return response()->download($path)->deleteFileAfterSend(true); // Delete the file after download
        }

        return redirect()->back()->with('error', 'File not found.');
    }

    public function exportErrorData()
    {
        return Excel::download(new ErrorDataExport($this->data_error), 'error_data.xlsx');
    }

}
