<?php

namespace App\Imports;

use App\Models\Item;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Validators\Failure;
use Illuminate\Support\Facades\Log;

class ItemImport implements ToModel, WithHeadingRow, WithValidation, SkipsOnFailure
{
    use SkipsFailures;

    private $successCount = 0;
    private $errors = [];
    private $errors_2 = [];

    public function model(array $row)
    {
        try {
            Log::info('Row data: ', $row);
            if ($this->isHeaderRow($row)) {
                return $this->processHeaderRow($row);
            }
        } catch (\Exception $e) {
            $this->errors_2[] = $e->getMessage();
            Log::error('Exception occurred: ', ['message' => $e->getMessage()]);
        }
    }

    private function isHeaderRow(array $row): bool
    {
        $expectedColumnsHeader = [
            'unique_number',
            'part_number',
            'address',
        ];
        return count(array_diff($expectedColumnsHeader, array_keys($row))) === 0 && !empty(array_filter($row));
    }

    private function processHeaderRow(array $row)
    {
        Log::info('Processing header data', ['row' => $row]);

        $validator = Validator::make($row, [
            'unique_number' => 'required',
            'part_number' => 'required',
            'address' => 'required',
        ]);
        
        try {
            if (Item::where('code', $row['unique_number'])->exists()) {
                $errorMessage = "The code {$row['unique_number']} already exists.";
                $this->errors_2[] = [
                    'unique_number' => $row['unique_number'],
                    'part_number' => $row['part_number'],
                    'address' => $row['address'],
                    'error_message' => $errorMessage
                ];
                $this->errors[] = $errorMessage;
                return null;
            }

            $item = new Item([
                'code' => $row['unique_number'],
                'name' => $row['part_number'],
                'address' => $row['address'],
                'created_who' => Auth::user()->name,
                // 'updated_who' => Auth::user()->name,
            ]);

            $item->save();
            $this->successCount++;
        } catch (\Exception $e) {
            $errorMessage = "The code {$row['unique_number']} already exists.";
            $this->errors_2[] = [
                'unique_number' => $row['unique_number'],
                'part_number' => $row['part_number'],
                'address' => $row['address'],
                'error_message' => $errorMessage
            ];
            $this->errors[] = $errorMessage;
        }
    }

    public function rules(): array
    {
        return [
            '*.unique_number' => 'required',
            '*.part_number' => 'required',
            '*.address' => 'required',
        ];
    }

    public function onFailure(Failure ...$failures)
    {
        foreach ($failures as $failure) {
            $this->errors[] = $failure->errors();
        }
    }

    public function getSuccessCount()
    {
        return $this->successCount;
    }

    public function getErrors()
    {
        return $this->errors;
    }

    public function getDataErrors()
    {
        return $this->errors_2;
    }
}
