<?php

namespace App\Imports;

use App\Models\Cases;
use App\Models\Destination;
use App\Models\LotProcess;
use App\Models\Item;
use App\Models\Mapping;
use App\Models\MappingDetail;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Maatwebsite\Excel\Validators\Failure;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class MappingImport implements ToModel, WithHeadingRow, SkipsOnFailure
{
    use SkipsFailures;

    private $message_error = [];
    private $data_error = [];
    private $dataErrorDetails = [];
    private $headers = [];
    private $successCount = 0;
    private $errorMessageDetails = [];

    public function model(array $row)
    {
        try {
            Log::info('Row data: ', $row);

            if ($this->isHeaderRow($row)) {
                return $this->processHeaderRow($row);
            } elseif ($this->isDetailRow($row)) {
                return $this->processDetailRow($row);
            }
        } catch (\Exception $e) {
            $this->message_error[] = $e->getMessage();
            Log::error('Exception occurred: ', ['message' => $e->getMessage()]);
        }
    }

    private function isHeaderRow(array $row): bool
    {
        $expectedColumnsHeader = [
            'destination',
            'lot_process',
            'case',
            'start_date',
            'end_date',
        ];
        return count(array_diff($expectedColumnsHeader, array_keys($row))) === 0 && !empty(array_filter($row));
    }

    private function isDetailRow(array $row): bool
    {
        $expectedColumnsDetail = [
            'sequence',
            'unique_number',
            'part_number',
            'address',
            'header_line',
        ];
        return count(array_diff($expectedColumnsDetail, array_keys($row))) === 0 && !empty(array_filter($row));
    }

    private function processHeaderRow(array $row)
    {
        Log::info('Processing header data', ['row' => $row]);

        $validator = Validator::make($row, [
            'destination' => 'required',
            'lot_process' => 'required',
            'case' => 'required',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
        ]);

        $destination = Destination::where('code', $row['destination'])->first();
        $lotProcess = LotProcess::where('code', $row['lot_process'])->first();
        $case = Cases::where('code', $row['case'])->first();

        $errorMessage = $this->validateHeader($validator, $destination, $lotProcess, $case, $row);

        if (!empty($errorMessage)) {
            $this->data_error[] = [
                'destination' => $row['destination'],
                'lot_process' => $row['lot_process'],
                'case' => $row['case'],
                'start_date' => $row['start_date'],
                'end_date' => $row['end_date'],
                'error_message' => $errorMessage,
            ];
            return null; // Skip saving if there are errors
        }

        $mapping = Mapping::create([
            'destination_id' => $destination->id,
            'lot_process_id' => $lotProcess->id,
            'case_id' => $case->id,
            'start_date' => $row['start_date'],
            'end_date' => $row['end_date'],
            'created_who' => Auth::user()->name,
        ]);
        $this->successCount++;
        $this->headers[] = $mapping->id;
        Log::info('Header saved:', ['mapping_id' => $mapping->id]);
        return $mapping;
    }

    private function validateHeader($validator, $destination, $lotProcess, $case, $row)
    {
        $errorMessage = [];

        if ($validator->fails()) {
            $errorMessage = array_merge($errorMessage, $validator->errors()->toArray());
        }
        if (!$destination) {
            $errorMessage[] = "Destination Code Not Available";
        }
        if (!$lotProcess) {
            $errorMessage[] = "Lot Process Code Not Available";
        }
        if (!$case) {
            $errorMessage[] = "Case Code Not Available";
        }
        if ($destination && $lotProcess && $case) {
            $isDuplicateHeader = Mapping::where('destination_id', $destination->id)
                ->where('lot_process_id', $lotProcess->id)
                ->where('case_id', $case->id)
                ->where('start_date', $row['start_date'])
                ->where('end_date', $row['end_date'])
 		->whereNotNull('deleted_at')
                ->exists();

            if ($isDuplicateHeader) {
                $errorMessage[] = "Headers with this combination and start and end dates already exist.";
            }
        }

        return $errorMessage;
    }

    private function processDetailRow(array $row)
    {
        Log::info('Processing detail data', ['row' => $row]);

        $validator = Validator::make($row, [
            'sequence' => 'required|integer',
            'unique_number' => 'required',
            'part_number' => 'required',
            'address' => 'required',
            'header_line' => 'required|integer',
        ]);

        if ($validator->fails()) {
            $this->dataErrorDetails[] = [
                'sequence' => $row['sequence'],
                'unique_number' => $row['unique_number'],
                'part_number' => $row['part_number'],
                'address' => $row['address'],
                'header_line' => $row['header_line'],
                'error_message' => $validator->errors()->toArray(),
            ];
            return null; // Skip saving if there are errors
        }

        $index = $row['header_line'] - 1;
        Log::info('Header line index:', ['index' => $index]);

        if (!isset($this->headers[$index])) {
            $errorMessage = "Header Error: Mapping ID not found for detail row with index {$row['header_line']}.";
            Log::error($errorMessage);

            // Add to dataErrorDetails with custom error message
            $this->dataErrorDetails[] = [
                'sequence' => $row['sequence'],
                'unique_number' => $row['unique_number'],
                'part_number' => $row['part_number'],
                'address' => $row['address'],
                'header_line' => $row['header_line'],
                'error_message' => ['header' => [$errorMessage]],
            ];
        }

        $mappingId = $this->headers[$index];
        $item = Item::where('code', $row['unique_number'])
            ->where('name', $row['part_number'])
            ->where('address', $row['address'])
            ->first();

        $errorMessageDetails = [];

        if (!$item) {
            $errorMessage = "Unique Number, Part Number, and Address combination not available.";
        }

        // Check for duplicate sequence if mapping ID is valid
        if ($mappingId) {
            $isDuplicateDetail = MappingDetail::where('mapping_id', $mappingId)
                ->where('sequence', $row['sequence'])
                ->exists();

            if ($isDuplicateDetail) {
                $errorMessageDetails[] = "The sequence already exists for this Mapping ID.";
            }
        }

        // Add errors if any
        if (!empty($errorMessageDetails)) {
            $this->dataErrorDetails[] = [
                'sequence' => $row['sequence'],
                'unique_number' => $row['unique_number'],
                'part_number' => $row['header_line'],
                'address' => $row['header_line'],
                'header_line' => $row['header_line'],
                'error_message' => $errorMessageDetails,
            ];
            return null; // Skip saving if there are errors
        }

        // Save detail mapping to database
        $detail = MappingDetail::create([
            'mapping_id' => $mappingId,
            'sequence' => $row['sequence'],
            'unique_number' => $row['unique_number'],
            'item_id' => $item->id,
	     'image' =>'images/mapping/1721228785_default.jpg',
            'created_who' => Auth::user()->name,
        ]);

        Log::info('Detail saved:', ['mapping_detail_id' => $detail->id]);
        return $detail;
    }

    public function onFailure(Failure ...$failures)
    {
        foreach ($failures as $failure) {
            $this->data_error[] = $failure->data();
        }
    }

    public function getSuccessCount()
    {
        return $this->successCount;
    }

    public function getDataErrors()
    {
        return $this->data_error;
    }

    public function getErrors()
    {
        return $this->message_error;
    }


    public function getDataErrorDetails()
    {
        return $this->dataErrorDetails;
    }

    public function getErrorMessageDetails()
    {
        return $this->errorMessageDetails;
    }
}
